<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Symfony\Component\HttpFoundation\StreamedResponse;

class BackupController extends Controller
{
    public function export()
    {
        $tables = [
            'product_types',
            'formulas',
            'calculations'
        ];

        $backupData = [];

        foreach ($tables as $table) {
            $backupData[$table] = DB::table($table)->get();
        }

        $json = json_encode($backupData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        $fileName = 'backup-' . date('Y-m-d-H-i-s') . '.json';

        Storage::disk('local')->put('backups/' . $fileName, $json);

        return response()->streamDownload(function () use ($json) {
            echo $json;
        }, $fileName, [
            'Content-Type' => 'application/json',
        ]);
    }

    public function exportCalculations()
    {
        $calculations = DB::table('calculations')
            ->join('product_types', 'calculations.product_type_id', '=', 'product_types.id')
            ->select('calculations.*', 'product_types.name as product_type_name')
            ->get();

        $fileName = 'calculations-' . date('Y-m-d-H-i-s') . '.json';

        return response()->streamDownload(function () use ($calculations) {
            echo json_encode($calculations, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        }, $fileName, [
            'Content-Type' => 'application/json',
        ]);
    }

    public function import(Request $request)
    {
        $request->validate([
            'backup_file' => 'required|file|mimes:json,sql|max:10240'
        ]);

        $file = $request->file('backup_file');
        $content = file_get_contents($file->getRealPath());
        $data = json_decode($content, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            return back()->with('error', 'ملف النسخ الاحتياطي غير صالح');
        }

        // البدء في معاملة قاعدة البيانات
        DB::beginTransaction();

        try {
            // حذف البيانات الحالية
            DB::table('calculations')->delete();
            DB::table('formulas')->delete();
            DB::table('product_types')->delete();

            // استعادة البيانات
            foreach ($data as $table => $records) {
                foreach ($records as $record) {
                    DB::table($table)->insert((array) $record);
                }
            }

            DB::commit();

            return back()->with('success', 'تم استعادة البيانات بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'فشل استعادة البيانات: ' . $e->getMessage());
        }
    }

    public function listBackups()
    {
        $files = Storage::disk('local')->files('backups');
        $backups = [];

        foreach ($files as $file) {
            $backups[] = [
                'name' => basename($file),
                'size' => Storage::disk('local')->size($file),
                'date' => Storage::disk('local')->lastModified($file)
            ];
        }

        // ترتيب تنازلي حسب التاريخ
        usort($backups, function ($a, $b) {
            return $b['date'] <=> $a['date'];
        });

        return $backups;
    }
}
