<?php

namespace App\Http\Controllers;

use App\Models\Calculation;
use App\Models\ProductType;
use App\Models\Formula;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class CalculationController extends Controller
{
    public function index()
    {
        $calculations = Calculation::with('productType')->latest()->paginate(10);
        return view('calculations.index', compact('calculations'));
    }

    public function create(Request $request)
    {
        $productTypes = ProductType::all();
        $duplicateId = $request->query('duplicate');
        $calculationToDuplicate = null;

        if ($duplicateId) {
            $calculationToDuplicate = Calculation::with('productType')->find($duplicateId);
        }

        return view('calculations.create', compact(
            'productTypes',
            'calculationToDuplicate'
        ));
    }

    public function store(Request $request)
    {
        // التحقق من البيانات
        $validator = Validator::make($request->all(), [
            'product_type_id' => 'required|exists:product_types,id',
            'title' => 'required|string|max:255',
            'variables' => 'required|array',
            'variables.*' => 'required|numeric',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        // الحصول على المعادلات
        $formulas = Formula::where('product_type_id', $request->product_type_id)
            ->where('is_active', true)
            ->orderBy('order')
            ->get();

        $calculatedResults = [];

        foreach ($formulas as $formula) {
            $result = $this->evaluateFormula($formula->formula, $request->variables);
            $calculatedResults[$formula->variable_name] = [
                'name' => $formula->name,
                'value' => $result,
                'unit' => $formula->unit,
                'formula' => $formula->formula
            ];
        }

        // إنشاء الحساب
        $calculation = Calculation::create([
            'product_type_id' => $request->product_type_id,
            'title' => $request->title,
            'input_variables' => $request->variables,
            'calculated_results' => $calculatedResults,
            'notes' => $request->notes
        ]);

        return response()->json([
            'success' => true,
            'calculation' => $calculation,
            'redirect' => route('calculations.show', $calculation->id)
        ]);
    }

    public function show(Calculation $calculation)
    {
        return view('calculations.show', compact('calculation'));
    }

    public function destroy(Calculation $calculation)
    {
        $calculation->delete();
        return redirect()->route('calculations.index')
            ->with('success', 'تم حذف الحساب بنجاح');
    }

    private function evaluateFormula($formula, $variables)
    {
        // استبدال المتغيرات بالقيم
        foreach ($variables as $key => $value) {
            // استبدال المتغيرات في المعادلة مثل {الطول} بقيمتها
            $formula = str_replace("{{$key}}", $value, $formula);
        }

        // تنظيف المعادلة - إزالة أي متغيرات متبقية
        $formula = preg_replace('/\{.*?\}/', '0', $formula);

        // التحقق من أن المعادلة تحتوي فقط على أرقام ورموز رياضية
        if (!preg_match('/^[0-9\+\-\*\/\(\)\.\s]+$/', $formula)) {
            \Log::error('معادلة غير صالحة تحتوي على أحرف غير مسموح بها: ' . $formula);
            return 0;
        }

        // تقييم المعادلة الرياضية
        try {
            // استخدام eval مع معالجة الأخطاء
            $result = @eval("return $formula;");

            // إذا فشل eval، حاول استخدام طريقة بديلة
            if ($result === false || $result === null) {
                // طريقة بديلة للتقييم
                $result = $this->safeEval($formula);
            }

            return round($result, 2);
        } catch (\Exception $e) {
            \Log::error('خطأ في تقييم المعادلة: ' . $e->getMessage(), [
                'formula' => $formula,
                'variables' => $variables
            ]);
            return 0;
        }
    }

    // دالة آمنة لتقييم المعادلات
    private function safeEval($formula)
    {
        // إزالة المسافات
        $formula = str_replace(' ', '', $formula);

        // معالجة الضرب والقسمة أولاً
        while (preg_match('/(\d+\.?\d*)([\*\/])(\d+\.?\d*)/', $formula, $matches)) {
            $num1 = floatval($matches[1]);
            $num2 = floatval($matches[3]);
            $operator = $matches[2];

            if ($operator === '*') {
                $result = $num1 * $num2;
            } else {
                if ($num2 == 0) {
                    $result = 0;
                } else {
                    $result = $num1 / $num2;
                }
            }

            $formula = str_replace($matches[0], $result, $formula);
        }

        // معالجة الجمع والطرح
        while (preg_match('/(\d+\.?\d*)([\+\-])(\d+\.?\d*)/', $formula, $matches)) {
            $num1 = floatval($matches[1]);
            $num2 = floatval($matches[3]);
            $operator = $matches[2];

            if ($operator === '+') {
                $result = $num1 + $num2;
            } else {
                $result = $num1 - $num2;
            }

            $formula = str_replace($matches[0], $result, $formula);
        }

        return floatval($formula);
    }
   
}
